from datetime import timedelta
import pandas as pd
import plotly.graph_objects as plt
import streamlit as st

from stock_sql.query_platform import pro_bar, stock_company, stk_managers, stk_rewards, pro, query_trade_cal
from stock_sql.query_sql import query_stock_info, query_keyword_like
from tool.stock_utils import convert_date_format, denifi_YYYYMMDD


@st.cache_data
def stock_code(code: str):
    pd_data = pro_bar(code)
    return pd_data


@st.cache_data
def all_code():
    return query_stock_info()


stock_info = all_code()
option = st.sidebar.selectbox(
    "choose stock code",
    stock_info["ts_code"],
    index=0,
    placeholder="Select stock code...",
)
st.sidebar.write(option)

options = st.sidebar.multiselect(
    "What are your var show",
    [['amount', "y2", "red"], ['vol', 'y3', "green"], ["am/vol", 'y4', 'orange'], ["pct_chg", 'y5', 'violet']],
    [],
)
st.sidebar.write(options)
# pd_data = stock_code("sh.000001")
pd_data = stock_code(option)

pd_data['amount'] = pd.to_numeric(pd_data['amount'], errors='coerce')
pd_data['vol'] = pd.to_numeric(pd_data['vol'], errors='coerce')
pd_data["am/vol"] = pd_data['amount'] / pd_data["vol"]  # 额度/数量
pd_data['trade_date'] = pd.to_datetime(pd_data['trade_date'])
pd_data['trade_date'] = pd_data['trade_date'].dt.date
start_time = pd_data['trade_date'].min()
end_time = pd_data['trade_date'].max()
# time slider
time_slider = st.slider(
    "Select a date range",
    value=(pd_data['trade_date'].min(), pd_data['trade_date'].max()),
    step=timedelta(days=1),
    format="MM/DD/YYYY"
)

st.write("Start time:", time_slider)

# selected_data = pd_data[
#     (pd_data['trade_date'] >= pd.to_datetime(time_slider)[0]) & (pd_data['trade_date'] <= pd.to_datetime(time_slider)[1])]

selected_data = pd_data[
    (pd_data['trade_date'] >= time_slider[0]) & (pd_data['trade_date'] <= time_slider[1])]

fig = plt.Figure()
# Add a line trace

fig.add_trace(plt.Scatter(x=selected_data['trade_date'], y=selected_data['close'], line=dict(color='black')))
fig.update_layout(
    xaxis=dict(
        tickformat='%Y-%m-%d',
        rangeselector=dict(
            buttons=list([
                dict(count=1, label="1d", step="day", stepmode="backward"),
                dict(count=7, label="1w", step="day", stepmode="backward"),
                dict(count=1, label="1m", step="month", stepmode="backward"),
                dict(count=1, label="1y", step="year", stepmode="backward"),
                dict(step="all")
            ])
        ),
        rangeslider=dict(visible=True),
        type='date',
        nticks=42,

    ),
    height=600
)

for index, item in enumerate(options):
    fig.add_trace(
        plt.Scatter(x=selected_data['trade_date'], y=selected_data[item[0]], name=item[0], line=dict(color=item[2]),
                    yaxis=f"y{index + 2}"))
    if index == 0:
        fig.update_layout(yaxis2=dict(title=item[0], side='right', overlaying='y', showgrid=False), )
    if index == 1:
        fig.update_layout(yaxis3=dict(title=item[0], side='right', overlaying='y', showgrid=False), )
    if index == 2:
        fig.update_layout(yaxis4=dict(title=item[0], side='right', overlaying='y', showgrid=False), )
    if index == 3:
        fig.update_layout(yaxis5=dict(title=item[0], side='right', overlaying='y', showgrid=False), )
    if index == 4:
        fig.update_layout(yaxis6=dict(title=item[0], side='right', overlaying='y', showgrid=False), )

# fig.add_trace(
#     plt.Scatter(x=selected_data['trade_date'], y=selected_data['amount'], name='Amount', line=dict(color='red'),
#                 yaxis='y2'))
# fig.update_layout(
#     yaxis2=dict(title='Amount', side='right', overlaying='y', showgrid=False),
# )
#
# fig.add_trace(
#     plt.Scatter(x=selected_data['trade_date'], y=selected_data['am/vol'], name='am/vol', line=dict(color='black'),
#                 yaxis='y3'))
# fig.update_layout(
#     yaxis3=dict(title='am/vol', side='right', overlaying='y', showgrid=False),
# )
st.markdown("### 走线图")
st.plotly_chart(fig, use_container_width=True, weight=1600, height=1600)

st.markdown("### 分钟行情")
select_day = st.selectbox(
    "选择交易日",
    query_trade_cal()["cal_date"].tolist(),
    index=0,
    placeholder="选择交易日",
)
trade_start = convert_date_format(select_day) + " 09:00:00"
trade_end = convert_date_format(select_day) + " 19:00:00"
pd_stk_mins = pro.stk_mins(ts_code=option, freq="1min", start_date=trade_start, end_date=trade_end)
# st.dataframe(pd_stk_mins)
st.line_chart(pd_stk_mins, x="trade_time", y=["close"])

# pd_company = stock_company(option)
# pd_company = pd_company.drop('uuid', axis=1)
# pd_company = pd_company.drop('cache_time', axis=1)
st.markdown("### 公司简讯")
st.dataframe(stock_company(option))
st.markdown("### 管理层人员")
st.dataframe(stk_managers(option))
st.markdown("### 管理层薪酬持股")
st.dataframe(stk_rewards(option))
st.markdown("### 资金流向")
st.dataframe(pro.moneyflow(ts_code=option))
st.markdown("### 分红")
st.dataframe(pro.dividend(ts_code=option))

st.markdown("### 个股资金流向")
pd_moneyflow = pro.moneyflow(ts_code=option, start_date=denifi_YYYYMMDD(1))
st.dataframe(pd_moneyflow)

st.markdown("### 融资融券交易额")
pd_slb_sec_detail = pro.slb_sec_detail(ts_code=option, start_date=denifi_YYYYMMDD(1))
st.dataframe(pd_slb_sec_detail)

st.markdown("### 转融券交易")
pd_margin_detail = pro.margin_detail(ts_code=option, start_date=denifi_YYYYMMDD(1))
st.dataframe(pd_margin_detail)

st.markdown("### 十大流通股东")
pd_top10_floatholders = pro.top10_floatholders(ts_code=option)
st.dataframe(pd_top10_floatholders)

st.markdown("### 股权质押明细")
pd_pledge_detail = pro.pledge_detail(ts_code=option)
st.dataframe(pd_pledge_detail)

st.markdown("### 股票回购")
pd_repurchase = pro.repurchase()
st.dataframe(pd_repurchase)

st.markdown("### 限售股解禁")
pd_share_float = pro.share_float(ts_code=option)
st.dataframe(pd_share_float)

st.markdown("### 大宗交易")
pd_block_trade = pro.block_trade(ts_code=option)
st.dataframe(pd_block_trade)

st.markdown("### 股东人数")
pd_stk_holdernumber = pro.stk_holdernumber(ts_code=option)
st.dataframe(pd_stk_holdernumber)

st.markdown("### 股东增减持")
pd_stk_holdertrade = pro.stk_holdertrade(ts_code=option)
st.dataframe(pd_stk_holdertrade)

st.markdown("### 利润表")
pd_income = pro.income(ts_code=option)
st.dataframe(pd_income)

st.markdown("### 资产负债表")
pd_balancesheet = pro.balancesheet(ts_code=option)
st.dataframe(pd_balancesheet)

st.markdown("### 现金流量表")
pd_cashflow = pro.cashflow(ts_code=option)
st.dataframe(pd_cashflow)

st.markdown("### 业绩快报")
pd_express = pro.express(ts_code=option)
st.dataframe(pd_express)

st.markdown("### 可转债信息")
pd_cb_basic = pro.cb_basic(ts_code=option)
st.dataframe(pd_cb_basic)

st.markdown("### 资讯")
stock_pd = pro.stock_basic(ts_code=option)
stock_name = stock_pd["name"].tolist()[0]
st.dataframe(query_keyword_like(keyword=stock_name))
print(stock_name)
